<?php

namespace Malicr;

class Manager{
    const db_prefix = 'malicr_';
    
    public static function run(){
        self::config();
        self::load();
        self::setup();
    }
    
    public static function config() {
        define( 'MALICR_URI', trailingslashit( plugin_dir_url( MALICR_FILE ) ) );
        define( 'MALICR_ASSETS_URI', MALICR_URI . 'assets/' );
        define( 'MALICR_LIB_PATH', MALICR_PATH . 'lib/' );
        define( 'MALICR_TEMPLATES_PATH', MALICR_PATH . 'templates/' );
    }
    
    public static function load() {
        require_once MALICR_LIB_PATH . 'class-shortcode.php';
        require_once MALICR_LIB_PATH . 'class-user-link.php';
        require_once MALICR_LIB_PATH . 'class-user.php';
        require_once MALICR_LIB_PATH . 'class-wp-list-table.php';
        require_once MALICR_LIB_PATH . 'class-wp-list-table-shortcodes.php';
        require_once MALICR_LIB_PATH . 'class-wp-list-table-user-shortcodes.php';
    }
    
    public static function setup() {
        register_activation_hook( MALICR_FILE, '\Malicr\Manager::activate' );
        add_action( 'admin_menu', '\Malicr\Manager::admin_menu' );
        add_action( 'admin_enqueue_scripts', '\Malicr\Manager::admin_scripts' );
        add_action( 'wp_ajax_malicr_add_shortcode', '\Malicr\Manager::ajax_add_shortcode' );
        add_action( 'wp_ajax_malicr_update_shortcode', '\Malicr\Manager::ajax_update_shortcode' );
        add_action( 'wp_ajax_malicr_delete_shortcode', '\Malicr\Manager::ajax_delete_shortcode' );
        add_action( 'wp_ajax_malicr_set_user_key', '\Malicr\Manager::ajax_set_user_key' );
        add_action( 'wp_ajax_malicr_update_user_links', '\Malicr\Manager::ajax_update_user_links' );
        add_shortcode( 'magiclink',  '\Malicr\Manager::shortcode_magiclink' );
        add_filter( 'the_content', '\Malicr\Manager::the_content' );
        add_action( 'wp', '\Malicr\Manager::set_cookies' );
        add_action( 'admin_init', '\Malicr\Manager::allow_subscriber_media_access' );
    }
    
    public static function allow_subscriber_media_access() {
        $contributor = get_role( 'subscriber' );
        $contributor->add_cap( 'upload_files' );
    }
    
# SHORTCODES
    
    public static function get_page_valid_key() {
        $key = filter_input( INPUT_GET, 'x' );
        if( false === $key || NULL === $key ) {
            return false;
        }
        
        $key = trim( $key );
        if( '' === $key ) {
            return false;
        }
        
        $user = \Malicr\User_Link::get_user_id_by_key( $key );
        if( false === $user ) {
            return false;
        }
        
        return $key;
    }
    
    public static function set_cookies() {
        $key = self::get_page_valid_key();
        if( false === $key ) {
            return;
        }
        setcookie( 'malicr_key', $key, time() + ( 86400 * 30 ), "/" ); // 30 days
    }
    public static function the_content( $content ) {
        $content = str_replace( 'http://[magiclink ', '[magiclink ', $content );

        return $content;
    }
    
    public static function shortcode_magiclink( $atts ) {
        $atts = shortcode_atts( array(
            'name' => ''
	), $atts, 'magiclink' );
        
        $name = isset( $atts['name'] ) ? trim( $atts['name'] ) : '';
        if( '' === $name ) {
            return '';
        }
        
        $shortcode = Shortcode::get_shortcode_by_name( $name );
        if( false === $shortcode ) {
            return '';
        }
        
        $user_id = false;
        $cookie_key = isset( $_COOKIE['malicr_key'] ) ? trim( $_COOKIE['malicr_key'] ) : false;
        if( false === $cookie_key ) {
            $cookie_key = self::get_page_valid_key(); // double check before cookies are properly set
        }
        if( false !== $cookie_key ) {
            $user_id = User_Link::get_user_id_by_key( $cookie_key );
        }
        
        if( false === $user_id){ // get default links
            return self::get_value_by_type( $shortcode->type, $shortcode->value );
        }
        
        $user_link = User_Link::get_by_shortcode_id( $shortcode->id, $user_id );
        if( '' === trim( $user_link->value ) ) { // link was not set by the user, load the default
            return self::get_value_by_type( $shortcode->type, $shortcode->value );
        }
        
        return self::get_value_by_type( $shortcode->type, $user_link->value );
    }
    
    public static function get_value_by_type( $type, $value ) {
        if( 'link' === $type ) {
            return $value;
        }
        if( 'text' === $type ) {
            return wpautop( $value );
        }
        
        // image
        $image = wp_get_attachment_image( $value, 'full' );
        return $image;
    }
    
    public static function get_image_url( $image_id, $size = 'thumbnail' ){
        $src = wp_get_attachment_image_src( $image_id, $size );
        if( empty( $src ) ) {
            return '';
        }
        
        return $src[0];
    }
    
# DATABASE
    
    public static function activate() {
        Shortcode::setup_db();
        User_Link::setup_db();
    }
    
# ADMIN PAGES
    
    public static function admin_scripts() {
        wp_enqueue_style( 'malicr-admin', MALICR_ASSETS_URI . 'css/admin.css' );
        wp_enqueue_script( 'malicr-admin', MALICR_ASSETS_URI . 'js/admin.js',
                ['jquery'], '1.0', true );
        $screen = get_current_screen();
        if( 'settings_page_malicr_settings' === $screen->id 
                || 'toplevel_page_malicr_user_settings' === $screen->id ) {
            if ( ! did_action( 'wp_enqueue_media' ) ) {
                wp_enqueue_media();
            }
        }
    }
    
    public static function admin_menu() {
        add_submenu_page( 'options-general.php', __( 'Magic Link Creator', 'malicr' ),
            __( 'Magic Link Creator', 'malicr' ), 'administrator', 'malicr_settings', 
            '\Malicr\Manager::admin_page_settings' );
        add_menu_page( __( 'Magic Link Creator', 'malicr' ), __( 'Magic Link Creator', 'malicr' ),
            'subscriber', 'malicr_user_settings', '\Malicr\Manager::admin_page_user_settings', 
            'dashicons-admin-links' );
    }
    
    public static function admin_page_settings() {
        ob_start();
        self::load_template( 'settings' );
        echo ob_get_clean();
    }
    
    public static function admin_page_user_settings() {
        ob_start();
        self::load_template( 'user-settings' );
        echo ob_get_clean();
    }
    
    
# AJAX
    
    public static function ajax_update_user_links() {
        if( ! current_user_can( 'subscriber' ) ) {
            self::ajax_error( __( 'Access denied.', 'malicr' ) );
        }
        
        $data_str = trim( filter_input( INPUT_POST, 'data' ) );
        if( '' === $data_str ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        parse_str( $data_str, $data );
        if( ! is_array( $data ) || empty( $data ) ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        $current_user_id = get_current_user_id();
        
        if( ! empty( $data['add'] ) ) {
            foreach( $data['add'] as $shortcode_id => $link ) {
                $link = trim( $link );
                $user_link = User_Link::get_by_shortcode_id( $shortcode_id, $current_user_id );
                if( false !== $user_link ) {
                    $user_link->update_link( $link );
                }
                else{
                    User_Link::insert_new( [
                        'user_id' => $current_user_id,
                        'shortcode_id' => $shortcode_id,
                        'value' => $link
                    ] );
                }
            }
        }
        if( ! empty( $data['update'] ) ) {
            foreach( $data['update'] as $id => $link ) {
                $link = trim( $link );
                $user_link = new User_Link( $id );
                if( ! $user_link->exists() ) {
                    continue;
                }
                $user_link->update_link( $link );
            }
        }
        
        self::ajax_success();
    }
    
    public static function ajax_set_user_key() {
        if( ! current_user_can( 'subscriber' ) ) {
            self::ajax_error( __( 'Access denied.', 'malicr' ) );
        }
        
        $key = trim( filter_input( INPUT_POST, 'key' ) );
        if( '' === $key ) {
            self::ajax_error( __( 'Please provide a valid key.', 'malicr' ) );
        }
        
        $user_id = User_Link::get_user_id_by_key( $key );
        $current_user_id = get_current_user_id();
        if( $user_id ) {
            if( $user_id === $current_user_id ) {
                self::ajax_success( '', [
                    'url' => add_query_arg( 'x', $key, home_url() )
                ] );
            }
            else{
                self::ajax_error( __( 'User with that key already exists, please choose a different one.', 'malicr' ) );
            }
        }
        
        if( ! self::is_valid_key( $key ) ) {
            self::ajax_error( __( 'Allowed characters for master key are letters, numbers and underscore ( _ ).', 'malicr' ) );
        }
        
        $set = User_Link::set_key_for_user( $current_user_id, $key );
        if( ! $set ) {
            self::ajax_error( __( 'Could not set the master key, please try again.', 'malicr' ) );
        }
        
        self::ajax_success( '', [
            'url' => add_query_arg( 'x', $key, home_url() )
        ] );
    }
    
    public static function ajax_add_shortcode() {
        if( ! current_user_can( 'administrator' ) ) {
            self::ajax_error( __( 'Access denied.', 'malicr' ) );
        }
        
        $data_str = trim( filter_input( INPUT_POST, 'data' ) );
        if( '' === $data_str ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        parse_str( $data_str, $data );
        if( ! is_array( $data ) || empty( $data ) ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        $data = wp_parse_args( $data, [
            'shortcode' => '',
            'value' => ''
        ] );
        
        $data['shortcode'] = trim( $data['shortcode'] );
        $data['value'] = trim( $data['value'] );
        if( '' === $data['shortcode'] || '' === $data['value'] ) {
            self::ajax_error( __( 'Please fill in the required fields.', 'malicr' ) );
        }
        
        if( ! self::is_valid_shortcode_name( $data['shortcode'] ) ) {
            self::ajax_error( __( 'Allowed characters for shortcode name are numbers, letters, underscore ( _ ) and dash ( - ).', 'malicr' ) );
        }
        
        if( false !== Shortcode::get_shortcode_by_name( $data['shortcode'] ) ) {
            self::ajax_error( __( 'Shortcode with that name already exists.', 'malicr' ) );
        }
        
        $type = 'link';
        if( ! empty( $data['type'] ) && in_array( $data['type'], ['link', 'text', 'image'] ) ) {
            $type = $data['type'];
        }
        
        $added = Shortcode::insert_new( [
            'name' => $data['shortcode'],
            'value' => $data['value'],
            'type' => $data['type']
        ] );

        if( ! $added ) {
            self::ajax_error( __( 'Shortcode could not be added. Please try again.', 'malicr' ) );
        }
        
        self::ajax_success();
    }
    
    public static function ajax_delete_shortcode() {
        if( ! current_user_can( 'administrator' ) ) {
            self::ajax_error( __( 'Access denied.', 'malicr' ) );
        }
        
        $id = (int) filter_input( INPUT_POST, 'id' );
        if( $id < 1 ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        $deleted = Shortcode::delete_by_id( $id );
        if( ! $deleted ) {
            self::ajax_error( __( 'Could not delete the shortcode. Please try again.', 'malicr' ) );
        }
        
        User_Link::delete_all_by_shortcode_id( $id );
        
        self::ajax_success();
    }
    
    public static function ajax_update_shortcode() {
        if( ! current_user_can( 'administrator' ) ) {
            self::ajax_error( __( 'Access denied.', 'malicr' ) );
        }
        
        $id = (int) filter_input( INPUT_POST, 'id' );
        $name = trim( filter_input( INPUT_POST, 'name' ) );
        $value = trim( filter_input( INPUT_POST, 'value' ) );
        
        if( $id < 1 ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        if( '' === $name || '' === $value ) {
            self::ajax_error( __( 'Shortcode name or value cannot be empty.', 'malicr' ) );
        }
        
        if( ! self::is_valid_shortcode_name( $name ) ) {
            self::ajax_error( __( 'Allowed characters for shortcode name are numbers, letters, underscore ( _ ) and dash ( - ).', 'malicr' ) );
        }
        
        $shortcode = new Shortcode( $id );
        if( ! $shortcode->exists() ) {
            self::ajax_error( __( 'Invalid request.', 'malicr' ) );
        }
        
        if( $name !== $shortcode->name && false !== Shortcode::get_shortcode_by_name( $name ) ) {
            self::ajax_error( __( 'Shortcode with that name already exists.', 'malicr' ) );
        }
        
        $updated = $shortcode->update( $name, $value );
        if( ! $updated ) {
            self::ajax_error( __( 'Could not update the shortcode. Please try again.', 'malicr' ) );
        }
        
        self::ajax_success();
    }
    
# COMMON
    
    public static function ajax_error( $message = '' ) {
        self::ajax_response( 'error', $message );
    }
    
    public static function ajax_success( $message = '', $data = [] ) {
        self::ajax_response( 'success', $message, $data );
    }
    
    public static function ajax_response( $type, $message = '', $data = [] ) {
        echo wp_json_encode( [
            'type'=> $type,
            'message' => $message,
            'data' => $data
        ] );
        exit;
    }
    
    public static function load_template( $template ) {
        include MALICR_TEMPLATES_PATH . $template . '.php';
    }
    
    public static function get_table_name( $table_name ) {
        global $wpdb;
        
        $db_prefix = self::db_prefix;
        $table_name = "{$wpdb->prefix}{$db_prefix}{$table_name}";
        
        return $table_name;
    }
    
    public static function is_valid_shortcode_name( $name ) {
        $matched = preg_match( '/^[a-zA-Z0-9_-]*$/', $name, $matches );
        
        return ( 1 === $matched );
    }
    
    public static function is_valid_key( $key ) {
        $matched = preg_match( '/^[a-zA-Z0-9_]*$/', $key, $matches );
        
        return ( 1 === $matched );
    }
}
